<?php

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Route;
use App\Http\Controllers\Api\AuthController;
use App\Http\Controllers\Api\VisitController;
use App\Http\Controllers\Api\VisitorRegistrationController;
use App\Http\Controllers\Api\LocationController;
use App\Http\Controllers\Api\DashboardController;
use App\Http\Controllers\Api\SettingsController;
use App\Http\Controllers\Api\TestAuthController;
use App\Http\Controllers\Api\PublicVisitController;

/*
|--------------------------------------------------------------------------
| API Routes
|--------------------------------------------------------------------------
|
| Here is where you can register API routes for your application. These
| routes are loaded by the RouteServiceProvider and all of them will
| be assigned to the "api" middleware group. Make something great!
|
*/

// ===========================
// TEST Authentication Routes (REMOVE IN PRODUCTION!)
// ===========================
Route::prefix('test')->group(function () {
    Route::get('/users', [TestAuthController::class, 'listTestUsers']);
    Route::post('/token', [TestAuthController::class, 'generateToken']);

    Route::middleware('auth:sanctum')->group(function () {
        Route::get('/me', [TestAuthController::class, 'me']);
        Route::post('/revoke', [TestAuthController::class, 'revokeToken']);
    });
});

// ===========================
// Authentication Routes
// ===========================
Route::prefix('auth')->group(function () {
    Route::get('/azure-ad/login', [AuthController::class, 'azureLogin']);
    Route::post('/azure-ad/callback', [AuthController::class, 'azureCallback']);

    // Protected routes
    Route::middleware('auth:sanctum')->group(function () {
        Route::get('/user', [AuthController::class, 'user']);
        Route::post('/logout', [AuthController::class, 'logout']);
    });
});

// ===========================
// Public Visitor Registration Routes
// ===========================
Route::prefix('visitor/registration')->group(function () {
    Route::get('/{id}/validate', [VisitorRegistrationController::class, 'validateToken']);
    Route::get('/{id}', [VisitorRegistrationController::class, 'getVisitDetails']);
    Route::post('/{id}/draft', [VisitorRegistrationController::class, 'saveDraft'])
        ->middleware('throttle:60,1'); // 60 attempts per 1 minute (for development)
    Route::post('/{id}/submit', [VisitorRegistrationController::class, 'submit'])
        ->middleware('throttle:60,1'); // 60 attempts per 1 minute (for development)
});

// ===========================
// Public Visit View Route
// ===========================
// NOTE: This route is deprecated - now using /visitor/registration routes instead
// Route::get('/public/visit/{id}', [PublicVisitController::class, 'show']);

// ===========================
// Protected API Routes
// ===========================
Route::middleware('auth:sanctum')->group(function () {

    // Locations (all authenticated users)
    Route::get('/locations', [LocationController::class, 'index']);

    // Visits
    Route::prefix('visits')->group(function () {
        Route::get('/', [VisitController::class, 'index']);
        Route::get('/{id}', [VisitController::class, 'show']);
        Route::post('/', [VisitController::class, 'store']);
        Route::put('/{id}', [VisitController::class, 'update']);
        Route::delete('/{id}', [VisitController::class, 'destroy']);

        // Visit Actions
        Route::post('/{id}/approve', [VisitController::class, 'approve'])
            ->middleware('role:security_member,security_admin,admin');
        Route::post('/{id}/reject', [VisitController::class, 'reject'])
            ->middleware('role:security_member,security_admin,admin');
        Route::post('/{id}/return', [VisitController::class, 'return'])
            ->middleware('role:security_member,security_admin,admin');
        Route::post('/{id}/send-reminder', [VisitController::class, 'sendReminder']);
        Route::get('/{id}/publish-link', [VisitController::class, 'getPublishLink']);
        Route::get('/{id}/tracking-history', [VisitController::class, 'trackingHistory']);

        // Export
        Route::get('/export', function () {
            // TODO: Implement export functionality
            return response()->json(['message' => 'Export feature coming soon']);
        });
    });

    // Dashboard
    Route::prefix('dashboard')->group(function () {
        Route::get('/stats', [DashboardController::class, 'stats']);
        Route::get('/recent-visits', [DashboardController::class, 'recentVisits']);
        Route::get('/today-visits', [DashboardController::class, 'todayVisits']);
        Route::get('/visit-status', [DashboardController::class, 'visitStatus']);
    });

    // Settings (Admin only)
    Route::prefix('settings')->middleware('role:admin')->group(function () {
        Route::get('/', [SettingsController::class, 'index']);
        Route::put('/azure-ad', [SettingsController::class, 'updateAzureAD']);
        Route::put('/dauha', [SettingsController::class, 'updateDauha']);
        Route::put('/smtp', [SettingsController::class, 'updateSMTP']);
        Route::put('/timezone', [SettingsController::class, 'updateTimezone']);

        // Email Templates
        Route::get('/email-templates', [SettingsController::class, 'emailTemplates']);
        Route::put('/email-templates/{id}', [SettingsController::class, 'updateEmailTemplate']);
    });
});

// Health check route
Route::get('/health', function () {
    return response()->json([
        'status' => 'ok',
        'timestamp' => now()->toIso8601String(),
    ]);
});
