<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use App\Models\Visit;
use App\Models\Setting;

class DauhaDSSService
{
    private $apiUrl;
    private $username;
    private $password;
    private $enabled;

    public function __construct()
    {
        $config = Setting::get('dauha_dss_config', []);
        $this->apiUrl = $config['apiUrl'] ?? null;
        $this->username = $config['username'] ?? null;
        $this->password = $config['password'] ?? null;
        $this->enabled = $config['enabled'] ?? false;
    }

    /**
     * Check if service is enabled
     */
    public function isEnabled(): bool
    {
        return $this->enabled && $this->apiUrl && $this->username && $this->password;
    }

    /**
     * Verify national ID with Dauha DSS
     */
    public function verifyNationalId(string $nationalId, string $name): bool
    {
        if (!$this->isEnabled()) {
            return true; // Skip verification if disabled
        }

        try {
            $response = Http::withBasicAuth($this->username, $this->password)
                ->timeout(30)
                ->post($this->apiUrl . '/verify-identity', [
                    'national_id' => $nationalId,
                    'name' => $name,
                ]);

            if ($response->successful()) {
                return $response->json('verified', false);
            }

            \Log::error('Dauha DSS verification failed', [
                'status' => $response->status(),
                'body' => $response->body(),
            ]);

            return false;
        } catch (\Exception $e) {
            \Log::error('Dauha DSS connection error: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Submit visit approval to Dauha DSS
     */
    public function submitVisitApproval(Visit $visit): bool
    {
        if (!$this->isEnabled()) {
            return true; // Skip if disabled
        }

        try {
            $mainVisitor = $visit->mainVisitor;

            $response = Http::withBasicAuth($this->username, $this->password)
                ->timeout(30)
                ->post($this->apiUrl . '/visit-approval', [
                    'request_id' => $visit->request_id,
                    'visitor_name' => $mainVisitor->name,
                    'visitor_email' => $mainVisitor->email,
                    'host_name' => $visit->host_name,
                    'location' => $visit->location->name,
                    'date' => $visit->date ?? $visit->date_from,
                    'status' => $visit->status,
                ]);

            if ($response->successful()) {
                return true;
            }

            \Log::error('Dauha DSS submission failed', [
                'status' => $response->status(),
                'body' => $response->body(),
            ]);

            return false;
        } catch (\Exception $e) {
            \Log::error('Dauha DSS submission error: ' . $e->getMessage());
            return false;
        }
    }
}
