# Beyti Visitor Management System - Laravel API Backend

## 🚀 Setup Instructions

### Prerequisites
- PHP 8.2+
- Composer
- MySQL 8.0+ or PostgreSQL 14+
- Node.js & NPM (for frontend integration)

### Installation Steps

#### 1. Generate Application Key
```bash
php artisan key:generate
```

#### 2. Create Database
Create a MySQL database named `beyti_visitor`:
```sql
CREATE DATABASE beyti_visitor CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
```

#### 3. Update Database Configuration
Update the `.env` file with your database credentials:
```env
DB_CONNECTION=mysql
DB_HOST=127.0.0.1
DB_PORT=3306
DB_DATABASE=beyti_visitor
DB_USERNAME=root
DB_PASSWORD=your_password
```

#### 4. Run Migrations
```bash
php artisan migrate
```

#### 5. Seed Database
```bash
php artisan db:seed
```

This will create:
- 3 sample locations (Cairo, Alexandria, Giza)
- 9 email templates
- Initial system settings

#### 6. Create Storage Link
```bash
php artisan storage:link
```

This creates a symbolic link from `public/storage` to `storage/app/public` for serving uploaded files.

#### 7. Configure File Permissions
```bash
chmod -R 775 storage bootstrap/cache
```

#### 8. Start Development Server
```bash
php artisan serve
```

The API will be available at: `http://localhost:8000`

---

## 📁 Project Structure

```
beyti-api/
├── app/
│   ├── Exceptions/           # Custom exception classes
│   ├── Http/
│   │   ├── Controllers/Api/  # API controllers
│   │   ├── Middleware/       # Custom middleware
│   │   └── Requests/         # Form request validators
│   ├── Jobs/                 # Queue jobs for emails
│   ├── Models/               # Eloquent models
│   ├── Policies/             # Authorization policies
│   └── Services/             # Business logic services
├── config/
│   ├── cors.php              # CORS configuration
│   └── sanctum.php           # Sanctum configuration
├── database/
│   ├── migrations/           # Database migrations
│   └── seeders/              # Database seeders
├── routes/
│   └── api.php               # API routes
└── storage/
    └── app/
        └── public/
            └── visitors/ids/ # Visitor ID images
```

---

## 🔑 API Endpoints

### Authentication
- `GET /api/auth/azure-ad/login` - Redirect to Azure AD login
- `POST /api/auth/azure-ad/callback` - Handle Azure AD callback
- `GET /api/auth/user` - Get current user (protected)
- `POST /api/auth/logout` - Logout (protected)

### Visits (Protected)
- `GET /api/visits` - List all visits
- `GET /api/visits/{id}` - Get visit details
- `POST /api/visits` - Create new visit
- `PUT /api/visits/{id}` - Update visit
- `DELETE /api/visits/{id}` - Cancel visit
- `POST /api/visits/{id}/approve` - Approve visit (Security only)
- `POST /api/visits/{id}/reject` - Reject visit (Security only)
- `POST /api/visits/{id}/return` - Return to host (Security only)
- `POST /api/visits/{id}/send-reminder` - Send reminder to visitor

### Visitor Registration (Public)
- `GET /api/visitor/registration/{token}/validate` - Validate token
- `GET /api/visitor/registration/{token}` - Get visit details
- `POST /api/visitor/registration/{token}/submit` - Submit registration

### Locations (Protected)
- `GET /api/locations` - Get active locations

### Dashboard (Protected)
- `GET /api/dashboard/stats` - Get statistics
- `GET /api/dashboard/recent-visits` - Get recent visits
- `GET /api/dashboard/today-visits` - Get today's visits

### Settings (Admin Only)
- `GET /api/settings` - Get all settings
- `PUT /api/settings/azure-ad` - Update Azure AD config
- `PUT /api/settings/smtp` - Update SMTP config
- `PUT /api/settings/dauha` - Update Dauha DSS config
- `GET /api/settings/email-templates` - Get email templates
- `PUT /api/settings/email-templates/{id}` - Update email template

### Health Check
- `GET /api/health` - API health check

---

## 🔐 Authentication

This system uses **Azure AD SSO** for authentication:

1. Frontend redirects to `/api/auth/azure-ad/login`
2. User authenticates with Microsoft credentials
3. Azure AD redirects to `/api/auth/azure-ad/callback`
4. Backend generates a Sanctum token
5. Frontend uses the token for subsequent API requests

**Header Format:**
```
Authorization: Bearer {token}
```

---

## 👥 User Roles

1. **admin** - Full system access
2. **user** - Can create and manage own visits
3. **security_member** - Can review and approve visits
4. **security_admin** - Security member + admin features

---

## 📧 Email System

The system sends 9 types of emails:

1. Visit created (to host & visitor)
2. After visitor submission (to security & host)
3. After approval (to visitor & host)
4. Rejection notice
5. Security return (for corrections)
6. Reminders

Email templates are customizable via the admin panel.

---

## 📊 Database Schema

### Main Tables
- `user_roles` - Email-based user roles
- `locations` - Office locations
- `visits` - Visit requests
- `visitors` - Visitor details with ID images
- `visit_status_history` - Audit trail
- `settings` - System configuration
- `email_templates` - Customizable email templates
- `personal_access_tokens` - Sanctum tokens (modified for email-based auth)

---

## 🔄 Visit Workflow

```
1. CREATE VISIT (Host)
   ↓
   Status: "Pending Visitor"
   ↓
2. VISITOR REGISTRATION (via token URL)
   ↓
   Status: "Pending Approve"
   ↓
3. SECURITY REVIEW
   ↓
   ├─ APPROVE → Status: "Approved"
   ├─ REJECT → Status: "Rejected"
   └─ RETURN → Status: "Pending Host"
```

---

## ⚙️ Configuration

### Azure AD Setup
1. Register application in Azure AD
2. Get Client ID, Tenant ID, and Client Secret
3. Set redirect URI: `http://your-domain.com/api/auth/azure-ad/callback`
4. Update configuration via admin panel or `.env`:

```env
AZURE_CLIENT_ID=your-client-id
AZURE_CLIENT_SECRET=your-client-secret
AZURE_TENANT_ID=your-tenant-id
AZURE_REDIRECT_URI=http://localhost:8000/api/auth/azure-ad/callback
```

### Email Configuration
Update SMTP settings via admin panel or `.env`:

```env
MAIL_MAILER=smtp
MAIL_HOST=smtp.gmail.com
MAIL_PORT=587
MAIL_USERNAME=your-email@gmail.com
MAIL_PASSWORD=your-app-password
MAIL_ENCRYPTION=tls
MAIL_FROM_ADDRESS=noreply@example.com
MAIL_FROM_NAME="Beyti Visitor Management"
```

---

## 🧪 Testing

### Create Test User
```php
use App\Models\UserRole;

UserRole::create([
    'email' => 'admin@example.com',
    'role' => 'admin',
    'is_active' => true,
]);
```

### Test API with Postman
1. Import the API collection (create one from routes)
2. Set base URL: `http://localhost:8000/api`
3. For protected routes, add header:
   ```
   Authorization: Bearer {your-token}
   ```

---

## 🚨 Troubleshooting

### Storage Permission Issues
```bash
chmod -R 775 storage
chmod -R 775 bootstrap/cache
```

### Database Connection Failed
- Check MySQL is running
- Verify database credentials in `.env`
- Ensure database exists

### CORS Issues
- Update `FRONTEND_URL` in `.env`
- Check `config/cors.php` settings

### Email Not Sending
- Verify SMTP credentials
- Check mail logs: `storage/logs/laravel.log`
- Test with Mailtrap or similar service

---

## 📝 Next Steps

1. **Configure Azure AD** - Set up Azure AD application
2. **Set up Email** - Configure SMTP settings
3. **Create Admin User** - Add first admin in `user_roles` table
4. **Customize Email Templates** - Update via admin panel
5. **Test Workflow** - Create test visit and go through complete flow
6. **Setup Queue Workers** - For production email processing:
   ```bash
   php artisan queue:work
   ```

---

## 🔒 Security Notes

- ✅ Email-based authentication (no passwords stored)
- ✅ Azure AD SSO integration
- ✅ Role-based access control (RBAC)
- ✅ File upload validation
- ✅ CORS protection
- ✅ Rate limiting on public endpoints
- ✅ Encrypted sensitive settings
- ✅ SQL injection prevention (Eloquent ORM)
- ✅ XSS prevention
- ✅ CSRF protection (for web routes)

---

## 📞 Support

For issues or questions, refer to:
- Laravel Documentation: https://laravel.com/docs
- Sanctum Documentation: https://laravel.com/docs/sanctum
- Project Documentation: `LARAVEL_API_DOCUMENTATION.md`

---

**Built with ❤️ for Beyti**
